<?php

namespace GameCMS\Services\VkApi;

use GameCMS\Clients\Curl;
use GameCMS\Common\Error;
use GameCMS\Services\VkApi\DTO\AuthorizeResponseDTO;

class Authorization
{
    private int $clientId;
    private string $apiVersion;
    private int $timeout;

    private string $codeChallengeMethod = 'S256';

    public function __construct(int $clientId, string $apiVersion, int $timeout = 10)
    {
        $this->clientId = $clientId;
        $this->apiVersion = $apiVersion;
        $this->timeout = $timeout;
    }

    public function getAuthorizationUrl(string $redirectUrl, string $data = ''): string
    {
        $codeVerifier = $this->generateCode(43);
        $state = $this->generateCode(32);

        $_SESSION['vk_auth_code_verifier'] = $codeVerifier;
        $_SESSION['vk_auth_state'] = $state;
        $_SESSION['vk_auth_data'] = $data;

        $params = [
            'code_challenge' => $this->getCodeChallenge($codeVerifier),
            'code_challenge_method' => $this->codeChallengeMethod,
            'client_id' => $this->clientId,
            'redirect_uri' => $redirectUrl,
            'response_type' => 'code',
            'state' => $state,
        ];

        return 'https://id.vk.ru/authorize?'.http_build_query($params);
    }

    /**
     * @param mixed $incomeCode
     * @param mixed $incomeState
     * @param mixed $incomeDeviceId
     * @param mixed $redirectUrl
     *
     * @return array{?AuthorizeResponseDTO, ?Error}
     */
    public function authorize($incomeCode, $incomeState, $incomeDeviceId, $redirectUrl): array
    {
        if (
            empty($_SESSION['vk_auth_code_verifier'])
            || empty($_SESSION['vk_auth_state'])
            || empty($_SESSION['vk_auth_data'])
        ) {
            return [null, error('empty session data')];
        }

        if ($incomeState !== $_SESSION['vk_auth_state']) {
            return [null, error('invalid income state')];
        }

        $codeVerifier = $_SESSION['vk_auth_code_verifier'];

        $params = [
            'grant_type' => 'authorization_code',
            'code_verifier' => $codeVerifier,
            'redirect_uri' => $redirectUrl,
            'code' => $incomeCode,
            'client_id' => $this->clientId,
            'device_id' => $incomeDeviceId,
            'state' => $incomeState,
        ];

        $curl = new Curl();
        $curl->setUrl('https://id.vk.ru');
        $curl->setTimeout($this->timeout);
        $response = $curl->post('/oauth2/auth', $params);

        $responseData = json_decode($response, true);
        if (!$responseData) {
            return [null, error('invalid response data')];
        }

        if (!empty($responseData['error'])) {
            return [null, error("{$responseData['error']}: {$responseData['error_description']}")];
        }

        if (empty($responseData['access_token'])) {
            return [null, error('invalid access token')];
        }

        if (empty($responseData['user_id'])) {
            return [null, error('invalid user id')];
        }

        if ($responseData['state'] !== $_SESSION['vk_auth_state']) {
            return [null, error('invalid access token state')];
        }

        $accessToken = $responseData['access_token'];
        $userId = $responseData['user_id'];
        $data = $_SESSION['vk_auth_data'];

        $usersService = new Users($accessToken, $this->apiVersion, $this->timeout);
        [$users, $error] = $usersService->getInfo([$userId]);
        if ($error) {
            return [null, error($error)];
        }

        if (!$users || !$users[0]) {
            return [null, error('cannot get user info')];
        }

        $userInfo = $users[0];

        $response = new AuthorizeResponseDTO();
        $response->user_id = $userInfo->user_id;
        $response->birth_day = $userInfo->birth_day;
        $response->first_name = $userInfo->first_name;
        $response->last_name = $userInfo->last_name;
        $response->photo = $userInfo->photo;
        $response->has_photo = $userInfo->has_photo;
        $response->data = $data;

        return [$response, null];
    }

    private function generateCode(int $length = 128): string
    {
        $chars = 'qazxswedcvfrtgbnhyujmkiolp1234567890QAZXSWEDCVFRTGBNHYUJMKIOLP';

        $code = '';
        while ($length--) {
            $code .= $chars[rand(0, strlen($chars) - 1)];
        }

        return $code;
    }

    private function getCodeChallenge(string $codeVerifier): string
    {
        return rtrim(strtr(base64_encode(hash('sha256', $codeVerifier, true)), '+/', '-_'), '=');
    }
}
